#!/tvbin/tivosh
#
# Copyright (C) 2003 - Brandon Hill
#
# This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License.
#
# This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; including all implied warranties.
#
# For more information, see the complete GPL at: www.gnu.org
#
# Revision History:
#
# Date     Vers Coments
# -------- ---- --------------------------------------------------
# 01/12/04 V1.0 Initial public release
# 01/13/04      Added v2.5.5 support
# 01/16/04 V1.1 Integrated all Version support, misc minor updates
# 10/30/04 V1.2 Additional error handling
# 12/02/05 V1.3 Additional database error handling
# 18/01/06 V1.4 Added v2.5.5 detection for offset values
# 17/12/09 V1.5 Rewrote space retrival procs + added large drive support (TivoZA)
# Sep 2011 [Mrt] changed /Recording/NowShowingByExpiration to /Recording/NowShowing otherwise it doesn't work on UK Tivos.
# 11/09/11 v1.6 [spitfires] 
#										Corrected recording calculation for UK TiVi (i.e. s/w v2.5.5)
#										Added "stop" as a synonym for "exit"									
#										Append to log file rather than create afresh each run (it's in /var/tmp so will be cleared on a reboot)
#										Added command-line parameters:  
#													-interval xxx   - Set cache rebuild interval secs (default 1800)
#													-noprogress  - Hide the "Getting Data" message
#													-nocounts      - Hide the recording counts
#													-notext          - Hide the "Now Playing" text
#													-viatc   			 - Refresh cache when going from Tivo Central to Now Playing (shortcutting the timeout)
#												(e.g. "autospace.tcl start -interval 300 -nocounts" )
#										Changed to retrieve data in the background by default  (the previous method can be achieved with the -viatc parameter)
#										(Hint: for a minimalist bar use "autospace.tcl start -nocounts -notext -noprogress" )
#										(Note: if you try to stop or restart the daemon when your TiVo is busy you may get an erroneous "ERROR: daemon is not active" message when in fact it is so active! Try again in 10 seconds time!)
#
# 14/08/2011 - [Mikerr] redbar theme added (merged from a previous version) 
#                       shows just the usage percentage bar in red
# 						           autospace.tcl start -redbar

set Prog            "AutoSpace"
set Vers            "V1.6"
set Priority        "fifo 1"

# OSD console fifo
set OSD             "/tmp/osdconsole"

set OutFile         "/tmp/autospace.log"
set BackGroundFlag  "BG"

# Control what is displayed on screen
set HideProgress  0					;# hide the "Getting Data..." message
set HideCounts  0						;# hide the recording counts (top right of display)
set HideText  0								;# hide the "Now Playing" message
set HidePercent  0						;# hide the percentage  overlay

# Re-get the data every x secs
set Reload_Interval 1800

# Screen context codes
set TivoCentral     2
set NowShowing      7
# here's some others for reference:
# set ToDo     32
# set MsgsandSetup     14
# set Messages     51
# set SysInfo     29

# Initialize Variables for Callback
set EventWait       0
set Restart         0
set Exit            0

set CurrentContext  0
set PreviousContext 0
set DataAge    0
global cons
set cons            0

set ViaTC 0											;# use change from Tivo Central to Now Playing to trigger extra data updates
set redbartheme 0  


#   Background                  Foreground
set Color(0) "\033\13341m"; set Color(6)  "\033\13331m"; # Overhead
set Color(1) "\033\13342m"; set Color(7)  "\033\13332m"; # Permanent
set Color(2) "\033\13343m"; set Color(8)  "\033\13333m"; # Recordings
set Color(3) "\033\13349m"; set Color(9)  "\033\13339m"; # Suggestions
set Color(4) "\033\13348m"; set Color(10) "\033\13338m"; # Deleted
set Color(5) "\033\13348m"; set Color(11) "\033\13338m"; # Free Space

# Set context init string
set    init "\033\133+p";        # Reset to initial state
append init "\033\1331;5;05;0y"; # Change Black
append init "\033\1331;5;11;1y"; # Change Red
append init "\033\1331;5;10;2y"; # Change Green
append init "\033\1331;5;12;3y"; # Change Yellow
append init "\033\1331;5;15;4y"; # Change Blue
append init "\033\1331;5;01;5y"; # Change Magenta
append init "\033\1331;5;10;6y"; # Change Cyan
append init "\033\1331;5;04;7y"; # Change White
append init "\033\1331;5;00;8y"; # Change Transparent
append init "\033\1331;5;14;9y"; # Change Charcoal Transparent
if {[file exists {/tvbin/zapPgdEtc.PAL.cs22}]} {
 set uktivo 1
 append init "\033\1331;2;7;54y"; # Change offsets (v2.5.5)
} else {
 set uktivo 0
 append init "\033\1331;2;7;46y"; # Change offsets (v3.0)
}
append init "\033\1331;4;2y";    # Change number of usable rows
append init "\033\1331;1y";      # Clear Page Buffer
append init "\033\13337;48m";    # Default colors

#
# Additional Event namespaces
#

namespace eval TmkEventII {
 variable EVT_USREVENT  98
}

namespace eval TmkUsrEvent {
 variable USR_PING    94
 variable USR_ACK     95
 variable USR_NACK    96
 variable USR_RESTART 97
 variable USR_EXIT    98
}

proc max {a b} {return [expr $a>$b?$a:$b]}

#
# Background error handler (print error and exit)
#

proc bgerror { error } {
 global errorInfo errorCode argv0 cons
 puts stderr "[clock format [clock seconds] -format "%D %T"] $argv0 FATAL: bgerror: $error"
 puts stderr "$errorInfo"
 puts stderr "$errorCode"
 if { $cons != 0 } {
  puts -nonewline $cons "\033\1331;1y\033\133+p"
  close $cons
 }
 exit 1
}

#
# Process events
#

proc EventCallback { type subtype } {
 global EventData
 switch -- $type \
  $TmkEvent::EVT_MW_STATUS {
   if { $subtype == $TmkEventMwStatus::CONTEXT_CHANGE } {
    global PreviousContext CurrentContext EventWait
    set PreviousContext $CurrentContext
    binary scan $EventData I CurrentContext
    if { $EventWait == -1 } { set EventWait $TmkEvent::EVT_MW_STATUS }
   }
  } \
	$TmkEvent::EVT_REMOTEEVENT {
		# keypress event handler
		# (watch out for the "3.0 event bug" if you are using TiVo s/w v3.0)	
		# note that many keypresses fire the handler twice, and also that a keypress will come in *before* any resulting context change event fires (obvious when you think about it!) so you might need to wait for a bit  (e.g.  "after 500; update;")
		if 1 { 
			binary scan $EventData I KeyPressed	
			puts stderr "Keypress $KeyPressed"
   }
  } \
  $TmkEventII::EVT_USREVENT {
   switch -- $subtype \
    $TmkUsrEvent::USR_RESTART {
     global Exit Restart EventWait
     set Exit 1
     set Restart 1
     if { $EventWait == -1 } { set EventWait $TmkEventII::EVT_USREVENT }
    } \
    $TmkUsrEvent::USR_EXIT {
     global Exit EventWait
     if { $Exit } {
      global cons
      if { $cons != 0 } {
       puts -nonewline $cons "\033\1331;1y\033\133+p"
       close $cons
      }
      exit 1
     }
     set Exit 1
     if { $EventWait == -1 } { set EventWait $TmkEventII::EVT_USREVENT }
    } \
    $TmkUsrEvent::USR_PING {
     event send $TmkEventII::EVT_USREVENT $TmkUsrEvent::USR_ACK .
    } \
    $TmkUsrEvent::USR_ACK  {
     global PingAck
     set PingAck 1
    } \
    $TmkUsrEvent::USR_NACK {
     global PingAck
     set PingAck 0
    }
  }
}

# #############################################################################################
#
proc ForeachMfsFile {idV nameV typeV mfsdir {prefix ""} {count 50} {body ""}} {
# Rewritten by BTUx9 to handle continue properly
  upvar $idV id
  upvar $nameV name
  upvar $typeV type
  set name $prefix
  set batch {}
  set prelen [string length $prefix]
  #~ set ispre $prelen
  incr prelen -1
  set blen $count
  set i $count
  while 1 {
    if [set code [catch {
      if {$i<$blen} {
        foreach {id name type} [lindex $batch $i] break
        if {$prelen>=0 && $prefix!=[string range $name 0 $prelen]} break
        incr i
        uplevel $body
      } else {RetryTransaction {
        if {$blen<$count} break
       	set batch [mfs scan $mfsdir -start $name -count $count]
        set blen [llength $batch]
        if {[lindex [lindex $batch 0] 1]==$name} { set i 1 } else { set i 0 }
      }}
    } errs]] {
        #~ puts "i=$i,code=$code,count=$count"
        if {$code<3} {
          if {$code==2} {return -code return $errs
          } elseif {[lsearch $::errorCode {errTmActiveLockConflict errTmBackgroundHoldoff errFsLockConflict}]>-1} {
            #~ puts "retrying after $::errorCode ..."
            after [random 200 2000]
            #~ continue
          } else { error $batch $::errorInfo $::errorCode }
        } elseif {$code==3} return
    }
  }
}

proc ForeachMfsFileTrans {idV nameV typeV mfsdir {prefix ""} {count 20} {read 1} {body ""}} {
# Rewritten by BTUx9 to work CORRECTLY... It required nested catch blocks, but it now:
## properly handles "continue" within the body
## retries intelligently (doesn't start from the beginning of the batch, unless read is 0)
## retries ONCE on unknown errors (most often caused by too many transactions) and drops the batch count
## Specify -noread when modifying MFS

  foreach v {id name type} {upvar [set ${v}V] $v }
  set name $prefix
  set batch {}
  set prelen [string length $prefix]
  #~ set ispre $prelen
  incr prelen -1
  set blen $count
  set i $count
  while 1 {
    set err $i
    set lasti $i
    if [set code [catch {RetryTransaction {
## Inside the inner while, you must use {return bReak} in order to exit the proc --btux9
      while {$i<$blen} {
        foreach {id name type} [lindex $batch $i] break
        if {$prelen>=0 && $prefix!=[string range $name 0 $prelen]} {return bReak}
        if [set code [catch {uplevel $body} errs]] {
            #~ puts "i=$i,code=$code"
          if {$code<3} {
            if {$code==2} {return $errs} else {error $batch $::errorInfo $::errorCode}
          } elseif {$code==3} {return bReak}
        }
        incr i
        set err 0
      }
      if {$blen<$count} break
     	set batch [mfs scan $mfsdir -start $name -count $count]
      set blen [llength $batch]
      if {[lindex [lindex $batch 0] 1]==$name} { set i 1 } else { set i 0 }
    }} errs]] {
      #~ puts "i=$i,code=$code,count=$count"
      if {$code>=3} return
      if {$code==2} { if {$errs=="bReak"} return else {return -code return $errs} }
      if {[lsearch $::errorCode {errTmActiveLockConflict errTmBackgroundHoldoff errFsLockConflict}]>-1} {
        #~ puts "retrying after $::errorCode ..."
        after [random 200 2000]
        #~ continue
      } elseif {$err!=$i} {
        set count [max [min [incr count -1] [expr $i-$err]] 5]
        #~ if {$count<1} {set count 1}
        puts "adjusting batchsize to $count"
      } else { error $batch $::errorInfo $::errorCode }
      if !$read { set i $lasti }
    }
  }
}

# #############################################################################################
#
proc get_space_tcl { } {
# A tcl only and mfs64 aware get_space function by jkozee.
# Adapted from code from Brandon Hill (autospace.tcl) and spike (mfslive 1.3).
  set tot_size 0
  set mfs64 0
  set devCnt 0

  set fd [open "/dev/hda10" RDONLY]; fconfigure $fd -translation binary

  # Determine if this is a mfs64 filesystem
  seek $fd 4; binary scan [read $fd 4] I magic
  if { [expr $magic & 0xffffffff] == 0xebbafeed } { set mfs64 1 }

  # Get the MFS device list
  if { $mfs64 } {
    seek $fd  36; set devlist [string trim [read $fd 132] "\000"]
    seek $fd 208; binary scan [read $fd 8] W pZone
  } else {
    seek $fd  36; set devlist [string trim [read $fd 128] "\000"]
    seek $fd 196; binary scan [read $fd 4] I pZone
  }
  close $fd

  # Loop through each MFS dev
  foreach {dev} $devlist {
    # Get the length for each partition
    scan "$dev" "%8s%d" part_name part_num
    set fd [open "$part_name" RDONLY]; fconfigure $fd -translation binary
    seek $fd [expr 512*$part_num+12]; binary scan [read $fd 4] I sec
    set devSectors($devCnt) [expr $sec & ~(1023)]
    close $fd

    # Get a handle for read only
    set fdDevs($devCnt) [open $dev RDONLY]
    fconfigure $fdDevs($devCnt) -translation binary
    incr devCnt
  }

  # Calculate MFS zone sizes for media regions
  while { $pZone != 0 } {
    # Determine the zone info
    for { set x 0 } { $x < $devCnt } { incr x } {
      if { $pZone < $devSectors($x) } { break }
      incr pZone [expr -$devSectors($x)]
    }

    # Determine the partion type
    if { $mfs64 } {
      seek $fdDevs($x) [expr 512*$pZone+92]
    } else {
      seek $fdDevs($x) [expr 512*$pZone+32]
    }
    binary scan [read $fdDevs($x) 4] I type

    # We are only interested in MFS media partions (type == 2)
    if { $type == 2 } {
      if { $mfs64 } {
        seek $fdDevs($x) [expr 512*$pZone+56]
        binary scan [read $fdDevs($x) 8] W size
      } else {
        seek $fdDevs($x) [expr 512*$pZone+52]
        binary scan [read $fdDevs($x) 4] I size
      }
      incr tot_size [expr $size/1024]
    }

    # Get the next zone ptr
    if { $mfs64 } {
      seek $fdDevs($x) [expr 512*$pZone+16]
      binary scan [read $fdDevs($x) 8] W pZone
    } else {
      seek $fdDevs($x) [expr 512*$pZone+12]
      binary scan [read $fdDevs($x) 4] I pZone
    }
  }

  # Close all open MFS Devices
  for { set x 0 } { $x < $devCnt } { incr x } { close $fdDevs($x) }

  return $tot_size
}

# #############################################################################################
#
proc getInProgress {} {
	# get size of the In Progress buffer
	# InProgress flag set as follows: h - Hidden recording	p - Permanent recording	r - normal recording	s - suggestion
  set db $::db
  set retl {}
  set bufpart [lindex {256 204} $::dtivo]
  ForeachMfsFile fsid name type "/Recording/LiveCache" "" 15 {
    set lc($fsid) 0
  }
  ForeachMfsFileTrans fsid name type "/Recording/InProgress" "" 15 1 {
    set rec [db $db openid $fsid]
    if [info exists lc($fsid)] {
      set parts [dbobj $rec gettarget Part]
      set size [expr [llength $parts] * $bufpart * 1024]

      set secs [expr round($size / (5960000.0/8/1024))]
      set code 100
    } else {
      set parts [dbobj $rec get Part]
      set size [expr [llength $parts] * 512 * 1024]
      set secs 0
      foreach part $parts {
        set begin [dbobj $part get Begin]
        set end [dbobj $part get End]
        if {$end != "" && $begin != ""} {
          set diff [expr ($end - $begin)/1000]
          incr secs $diff
        }
      }
      set code 101
    }
    lappend retl $code $size $secs

    set selectiontype [dbobj $rec get SelectionType]
    if { $selectiontype != 0 } {
     set expirationdate [dbobj $rec get ExpirationDate]
     if { [catch { set recordingbehavior [dbobj $rec get RecordingBehavior]}] } {
      set presentationbehavior $selectiontype
     } else {
      set presentationbehavior [dbobj $recordingbehavior get PresentationBehavior]
     }
     if       { $presentationbehavior == 8 } { lappend ::InProgress "h" } \
     elseif { $presentationbehavior == 6 } { lappend ::InProgress "s" } \
     elseif { $expirationdate == 24855   }   { lappend ::InProgress "p" } \
     else                                                      { lappend ::InProgress "r" }
    }
  }
  return $retl
}

# #############################################################################################
#
proc space_summary {} {
	# Calculate used space and recording counts
	#
	set ::Spc_Available 0; set ::Spc_Deleted  0; set ::Spc_Suggest  0; set ::Spc_Perm 0; set ::Spc_Recording 0; set ::Cnt_Deleted 0; set ::Cnt_Suggest 0; set ::Cnt_Perm 0; set ::Cnt_Recording 0; set ::InProgress { }
	
  set descs {99 Total 6 Deleted 5 Suggestions 2 Recordings 3 Expirable 4 {Keep Forever} 0 {Tivo Clips} -100 {Live Buffer} -101 Recording -102 Misc}
  foreach {n d} $descs { set k($n) 0 }
  foreach {n d} $descs { set count($n) 0 }
  if {$::version<4} {
		if {$::version<3} { set mfsdir "NowShowing" } else { set mfsdir "NowShowingByExpiration" }
    set mask {id sizek}	  
    ForeachMfsFile fsid name type "/Recording/$mfsdir" "" 55 {
      if [string match "*:24855:*" $name] { set v 4 } else { set v [string index $name 0] }
      set arr($fsid) $v
    }
		# [spitfires] - this breaks backwards compatability.  This just totals everything up and dumps it into "misc" - this is useless for Series 1 TiVos. So reinstate the v1.4 code:
		if {$::version>=3} {  						;# [spitfires] is this correct? (I can only test on v2.5.5)
			ForeachMfsFile fsid name type "/Recording/DiskUsed/10" "" 55 {
				if [info exists arr($fsid)] {
					foreach $mask [split $name :] break
					set v $arr($fsid)
					if [info exist k($v)] {
						incr k($v) [expr $sizek/1024]
						incr count($v) 1
					} else {
						set k($v) [expr $sizek/1024]
						set count($v) 1
					}
					unset arr($fsid)        
				}
			}
			set totk 0
			foreach {fsid v} [array get arr] { incr totk $v }
			set k(-102) $totk
		
		} else {		
			# ...reinstate the v1.4 code:
			ForeachMfsFile fsid name type "/Recording/DiskUsed/10" "" 55 {
				RetryTransaction {
					set recording            [db $::db openid $fsid]
					set state               		 [dbobj $recording get State]
					set selectiontype      [dbobj $recording get SelectionType]
					set streamfilesize     [dbobj $recording get StreamFileSize]
					set expirationdate    [dbobj $recording get ExpirationDate]
					if { [catch { set recordingbehavior [dbobj $recording get RecordingBehavior]}] } {
						set presentationbehavior $selectiontype
					} else {
						set presentationbehavior [dbobj $recordingbehavior get PresentationBehavior]
					}
				}
				# convert to MB as per Spc_Total
				set streamfilesizeMB [expr $streamfilesize / 1024]
				if     { $state == 5 }                                            	\
							{ incr count(6);  incr k(6) $streamfilesizeMB } 		\
				elseif { $presentationbehavior == 8 }                	\
							{ }                                                       				 	\
				elseif { $presentationbehavior == 6 }                  \
							{ incr count(5);   incr k(5) $streamfilesizeMB }		 \
				elseif { $expirationdate == 24855 }                    	 \
							{ incr count(4);   incr k(4) $streamfilesizeMB } 	 \
				else   { incr count(2); incr k(2) $streamfilesizeMB }
			}
		#
    }
		
    # TiVo's own videos
		ForeachMfsFile fsid name type "/Recording/DiskUsed/11" "" 55 {
      foreach $mask [split $name :] break
      incr k(0) [expr $sizek/1024]
    }
  } else {
    set mask {x cat sizek}
    set i 0
    ForeachMfsFile fsid name type /DiskUsed "011" 50 {
      foreach $mask [split $name :] break
      incr i [expr $sizek/1024]
    }
    set k(0) $i
    ForeachMfsFile fsid name type /DiskUsed "010" 50 {
      foreach $mask [split $name :] break
      set cat [expr 1$cat - 1000]
      if [info exists k($cat)] {
      	incr k($cat) [expr $sizek/1024]
      	incr count($cat) 1      	
      } else {
      	set k($cat) [expr $sizek/1024]
      	set count($cat) 1      	
      }
    }
  }
	
  # Get recording currently in progress
	if [info exists k(9)] {
    set k(-100) $k(9)
    unset k(9)
    set count(-100) $count(9)
    unset count(9)
  } else {
    foreach {type sizek secs} [getInProgress] {
      incr k(-$type) [expr $sizek/1024]
      incr count(-$type) 1
    }
  }

  set totk 0
  foreach {n v} [array get k] {incr totk $v}
	# debug=  foreach {n v} [array get k] { puts "$n = $v  [lindex $descs [expr [lsearch $descs $n]+1]]" }

  set ::Spc_Available [expr $::Spc_Total - $totk]
  set ::Spc_Deleted   $k(6)
  set ::Spc_Suggest   $k(5)
  set ::Spc_Perm      $k(4)
  set ::Spc_Recording $k(2)
  set ::Cnt_Deleted   $count(6)
  set ::Cnt_Suggest   $count(5)
  set ::Cnt_Perm      $count(4)
  set ::Cnt_Recording $count(2)
}


# #############################################################################################
#
proc buildosd {} {
	# Build the OnScreenDisplay

	global Spc_Total Spc_Available Spc_Deleted Spc_Suggest Spc_Perm Spc_Recording Cnt_Deleted  Cnt_Suggest Cnt_Perm Cnt_Recording
	global Color InProgress uktivo dat

  set Spc_Overhead [max [expr $Spc_Total-$Spc_Perm-$Spc_Recording-$Spc_Suggest-$Spc_Deleted-$Spc_Available] 0]

  # Convert to percentages (round to nearest 2% for display)
  set tot $Spc_Total
  set prm [expr round((50*($Spc_Perm     +0.0))/$tot)*2]
  set rec [expr round((50*($Spc_Recording+0.0))/$tot)*2]
  set sug [expr round((50*($Spc_Suggest  +0.0))/$tot)*2]
  set del [expr round((50*($Spc_Deleted  +0.0))/$tot)*2]
  set fre [expr round((50*($Spc_Available+0.0))/$tot)*2]
  set ovr [max [expr 100-$prm-$rec-$sug-$del-$fre] 0]

  # Start output string with defaults
  set dat "\033\1331;1y\033\13337;48m"

  # Build Space Graph (Overlay with % Used)
  append dat "\033\1332;6H\021"
  for { set x 0 } { $x < [expr $fre+$del+$sug+$rec+$prm+$ovr] } { incr x 2 } {
   set idx [expr [expr $x>=                    $ovr]+ \
                        [expr $x>=               $prm+$ovr]+ \
                        [expr $x>=          $rec+$prm+$ovr]+ \
                        [expr $x>=     $sug+$rec+$prm+$ovr]+ \
                        [expr $x>=$del+$sug+$rec+$prm+$ovr]]
   append dat "$Color([expr $idx+3*($x % 4)])"
   if { [expr $x % 4] != 0 } {
    append dat "\336\033\1333;4;9;37m\010 \033\13329m\010 \033\13323;24m"
   }
  }
  append dat "\033\1330;48m\020"


  if (!$::HidePercent) {
  	# Overlay percentage on bar graph
  	set    pct [expr round((100*($Spc_Overhead+$Spc_Perm+$Spc_Recording+0.0))/$tot)]
  	append dat "\033\1332;15H\033\1336m[format "%3d" $pct]% Used\033\13326m"
  	}
  # Build Statistics
  if (!$::HideText) {
		set    shw [expr $Cnt_Perm+$Cnt_Recording+$Cnt_Suggest]
		if {$uktivo} {
			append dat "\033\1331;6HNow Playing: $shw "
		} else {
			append dat "\033\1331;6HNow Showing: $shw "
		}
	} else { 				;# need to erase the "Getting Data.." message
		append dat "\033\1331;6H               "
	}

	if ($::redbartheme) { append dat "$Color(9)($Cnt_Suggest)" }
	if {!$::HideCounts} {
		if { [expr $Cnt_Perm+$Cnt_Recording+$Cnt_Suggest] > 0} {
		 set pos [expr 35-[string length "$Cnt_Perm $Cnt_Recording $Cnt_Suggest"]]
		 append dat "\033\1331;${pos}H\133"
		 append dat "$Color(7)$Cnt_Perm $Color(8)$Cnt_Recording $Color(9)$Cnt_Suggest"
		 append dat "\033\13337m\135"
		}
	}

  # Build InProgress Indicator
  if { [expr [llength $InProgress]] > 0 } {
   append dat "\033\1332;34H"
   foreach {type} $InProgress {
    if     {$type == "h"} {append dat "\033\13341m "} \
    elseif {$type == "s"} {append dat "\033\13349m "} \
    elseif {$type == "p"} {append dat "\033\13342m "} \
    else                  {append dat "\033\13343m "}
   }
   # Surround with red bars on top and bottom
   set Spaces [string range "   " 0 [expr [llength $InProgress]-1]]
   append dat "\033\1332;34H\033\1333;4;9;31m$Spaces"
   append dat "\033\1332;34H\033\13329m$Spaces\033\13323;24;37;48m"
  }
	 
	return
}


# #############################################################################################
#
# Start of main section
#

# Condense argument lists to a single string
set argv [join $argv]
set argc [llength $argv]

if { [lindex $argv [expr $argc-1]] != "$BackGroundFlag" } {

#
# Running in foreground
#

 if { [lindex $argv 0] != "start" } {
  # Register for internal events
  event register $TmkEventII::EVT_USREVENT  EventCallback

  # Check for existing daemon (Send USR_PING, check for USR_ACK / USR_NACK)
  event send $TmkEventII::EVT_USREVENT $TmkUsrEvent::USR_PING .
  set Timer [after 1000 EventCallback $TmkEventII::EVT_USREVENT $TmkUsrEvent::USR_NACK]
  vwait PingAck
  if { $Timer != 0 } { after cancel $Timer; set Timer 0 }
  set Status $PingAck
 } else {
  set Status 0
 }

 switch [lindex $argv 0] \
  "" - "start" {
   # If not already running, restart in background with background flag
   if { $Status } {
    puts stderr "FATAL: Cannot start, daemon is already active"
   } else {
    # Check for osdcons fifo
    if { !([file exists $OSD] && \
           [file type $OSD] == "fifo" && \
           ![catch { set cons [open $OSD {WRONLY NONBLOCK}] } ] ) } {
     if { [catch {set pid [exec osdcons >&/dev/null &]}] && \
          [catch {set pid [exec [file dirname [info script]]/osdcons >&/dev/null &]}] } {
      puts stderr "FATAL: osdcons not active and cannot be started ($OSD not a writable fifo)"
      exit 1
     } else {
      after 1000
      if { [catch {exec setpri $Priority $pid}] && \
           [catch {exec [file dirname [info script]]/setpri $Priority $pid}] } {
       set Priority "default"
      }
      if { !([file exists $OSD] && \
             [file type $OSD] == "fifo" && \
             ![catch { set cons [open $OSD {WRONLY NONBLOCK}] } ] ) } {
       puts stderr "FATAL: osdcons not active and cannot be started ($OSD not a writable fifo)"
       exit 1
      } else {
       puts stderr "INFO: osdcons started (Pid: $pid, Priority: $Priority)"
      }
     }
    }
    puts -nonewline $cons "\033\1331;1y\033\133+p"
    close $cons
    exec $argv0 $argv $BackGroundFlag >>&$OutFile &
    puts stderr "INFO: daemon started, output is in $OutFile"
   }
  } \
  "status" {
   if { $Status } {
    puts stderr "INFO: daemon is active"
   } else {
    puts stderr "INFO: daemon is not active"
   }
  } \
  "restart" {
   if { $Status } {
    event send $TmkEventII::EVT_USREVENT $TmkUsrEvent::USR_RESTART .
    puts stderr "INFO: restart event sent"
   } else {
    puts stderr "ERROR: daemon is not active"
   }
  } \
  "info" {
		#Open database
		set db [dbopen]

		RetryTransaction {
		  set swsystem [db $db open /SwSystem/ACTIVE]
		  set tivoswversion [dbobj $swsystem get Name]
		  set version [lindex [split $tivoswversion .] 0]    
		}

		set dtivo [MfsFileExists /ApgBoot]

		set Spc_Total [expr [get_space_tcl]/2]
	  space_summary
	  
	  puts stderr "Spc_Total: $::Spc_Total MB"
	  puts stderr "Spc_Available: $::Spc_Available MB"
	  puts stderr "Spc_Deleted: $::Spc_Deleted MB"
	  puts stderr "Spc_Suggest: $::Spc_Suggest MB" 
	  puts stderr "Spc_Perm: $::Spc_Perm MB"
	  puts stderr "Spc_Recording: $::Spc_Recording MB"
	  puts stderr "Cnt_Deleted: $::Cnt_Deleted"
	  puts stderr "Cnt_Suggest: $::Cnt_Suggest"
	  puts stderr "Cnt_Perm: $::Cnt_Perm"
	  puts stderr "Cnt_Recording: $::Cnt_Recording"
	  
	  # Close DataBase
		dbclose $db  	
  } \
  "stop" - "exit" {
   if { $Status } {
    event send $TmkEventII::EVT_USREVENT $TmkUsrEvent::USR_EXIT .
    puts stderr "INFO: exit event sent"
   } else {
    puts stderr "ERROR: daemon is not active"
   }
  } \
  default {
   puts stderr "Usage: $argv0"
   puts stderr " Args:        <none>  - Start daemon"
   puts stderr "              start   - Start daemon"
   puts stderr "              status  - Determine status of daemon"
   puts stderr "              restart - Terminate and restart daemon"
   puts stderr "              exit    - Safely terminate daemon"
   puts stderr "              stop    - Same as exit!"
	 puts stderr " Extra Args (only with 'start'):"
   puts stderr "              -interval    - Followed a number - set stats reload interval (secs)"
   puts stderr "              -noprogress  - Hide the Getting Data message"
   puts stderr "              -nocounts    - Hide the recording counts"
   puts stderr "              -notext      - Hide the Now Playing text"
   puts stderr "              -viatc   - Retrieve data when going from Tivo Central to Now Playing"
   puts stderr "              -redbar   - Bare theme showing only percentage bar in red"
  }
 exit
}

#
# Running in background
#

# Strip background flag from args
incr argc -1
set argv [lreplace $argv $argc $argc]

# Prcoess the parameters
for {set i 1} {$i <= $argc} {incr i} {
	switch -- [lindex $argv $i] {
		"-interval" { incr i; set Reload_Interval [lindex $argv $i]  }
		"-noprogress" { set HideProgress 1 }
		"-nocounts" { set HideCounts 1 }
		"-notext" { set HideText 1 }
		"-viatc" { set ViaTC 1 }
  	"-redbar" { 
		# Just want a red bar only, and no stats thanks

	 	set Color(0) "\033\13341m"; set Color(6)  "\033\13331m"; # Overhead
	 	set Color(1) "\033\13341m"; set Color(7)  "\033\13331m"; # Permanent
	 	set Color(2) "\033\13341m"; set Color(8)  "\033\13331m"; # Recordings

		set HidePercent 1
	  set HideCounts 1
	 	set redbartheme 1 }
		"" - "BG" {}
		default { puts stderr "Unknown argument - [lindex $argv $i]" }
	}
}

# Change our priority
if { [catch {exec setpri $Priority [pid]}] && \
     [catch {exec [file dirname [info script]]/setpri $Priority [pid]}] } {
 set Priority "default"
}

#Open database
set db [dbopen]

RetryTransaction {
  set swsystem [db $::db open /SwSystem/ACTIVE]
  set tivoswversion [dbobj $swsystem get Name]
  set version [lindex [split $tivoswversion .] 0]    
}

set dtivo [MfsFileExists /ApgBoot]

# Get total disc space
set Spc_Total [expr [get_space_tcl]/2]

# Open osdcons fifo
set cons [open $OSD {WRONLY NONBLOCK}]

# Register for  events
event register $TmkEvent::EVT_MW_STATUS  EventCallback
event register $TmkEventII::EVT_USREVENT EventCallback
# Keypress event handler
# #	event register $TmkEvent::EVT_REMOTEEVENT EventCallback


# We are ready to go
puts stderr "[clock format [clock seconds] -format "%D %T"] $Prog $Vers Started (Priority $Priority) \n Args: $argv"

# #############################################################################################
#
while {!$Exit} {
	set EventWait -1
	set Timer [after [expr $Reload_Interval *1000] "set EventWait -2"]
  vwait EventWait
  if { $Timer != 0 } { after cancel $Timer; set Timer 0 }

	if { $Exit || ($ViaTC && $CurrentContext != $NowShowing && $EventWait != -2)} { continue }

	if { ($ViaTC && $PreviousContext == $TivoCentral) || [expr $DataAge + $Reload_Interval] <= [clock seconds] } {
   
		if {$CurrentContext == $NowShowing} {
			# Reset the console
			puts -nonewline $cons $init
		
			if {!$HideProgress} {
				puts -nonewline $cons "\033\133H     Getting Data.\033\133J"
				flush $cons
			}
		}
	
		puts stderr "[clock format [clock seconds] -format "%T"] Getting data  (interval = $Reload_Interval secs)"; set etime [clock clicks]
		space_summary
		set DataAge [clock seconds]
		puts stderr "[clock format [clock seconds] -format "%T"] Data got  ([format {%.1f} [expr ([clock clicks]-$etime)/1e6]] secs)"
		
		update	
		if {$CurrentContext == $NowShowing && !$HideProgress} {
			puts -nonewline $cons ".\014"
			flush $cons
		}
	
		set dat ""
		buildosd
		
	}

	update; if { $Exit || $CurrentContext != $NowShowing } { continue }
	
	# Report data to user
	puts -nonewline $cons $init			;# reset in case it hasn't already been done
	puts -nonewline $cons "$dat\014"

	flush $cons

}

# Close OSD fifo
puts -nonewline $cons "\033\1331;1y\033\133+p"
close $cons

# Close DataBase
dbclose $::db

# We are done
puts stderr "[clock format [clock seconds] -format "%D %T"] $Prog $Vers Ending"

# Check for restart
if { $Restart } { exec $argv0 $argv $BackGroundFlag >>&$OutFile & }

# Exit program normally
exit
